package atomicstryker.ruins.common;

import java.io.File;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.tileentity.TileEntityCommandBlock;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.TextComponentTranslation;
import net.minecraft.world.World;
import net.minecraft.world.WorldProviderHell;
import net.minecraft.world.chunk.IChunkGenerator;
import net.minecraft.world.chunk.IChunkProvider;
import net.minecraft.world.chunk.storage.AnvilChunkLoader;
import net.minecraft.world.storage.ISaveHandler;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.util.FakePlayer;
import net.minecraftforge.event.entity.EntityEvent;
import net.minecraftforge.event.entity.player.PlayerEvent.BreakSpeed;
import net.minecraftforge.event.world.BlockEvent.BreakEvent;
import net.minecraftforge.event.world.WorldEvent;
import net.minecraftforge.fml.client.FMLClientHandler;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.common.IWorldGenerator;
import net.minecraftforge.fml.common.Mod;
import net.minecraftforge.fml.common.Mod.EventHandler;
import net.minecraftforge.fml.common.event.FMLInitializationEvent;
import net.minecraftforge.fml.common.event.FMLServerStartingEvent;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.common.network.NetworkCheckHandler;
import net.minecraftforge.fml.common.registry.GameRegistry;
import net.minecraftforge.fml.relauncher.Side;

@Mod(modid = "ruins", name = "Ruins Mod", version = RuinsMod.modversion, dependencies = "after:extrabiomes")
public class RuinsMod
{
    static final String modversion = "16.2";

    public final static int DIR_NORTH = 0, DIR_EAST = 1, DIR_SOUTH = 2, DIR_WEST = 3;
    public static final String BIOME_ANY = "generic";

    private ConcurrentHashMap<Integer, WorldHandle> generatorMap;
    private ConcurrentLinkedQueue<int[]> currentlyGenerating;

    @NetworkCheckHandler
    public boolean checkModLists(Map<String, String> modList, Side side)
    {
        return true;
    }

    @EventHandler
    public void load(FMLInitializationEvent evt)
    {
        generatorMap = new ConcurrentHashMap<>();
        currentlyGenerating = new ConcurrentLinkedQueue<>();
        GameRegistry.registerWorldGenerator(new RuinsWorldGenerator(), 0);
        MinecraftForge.EVENT_BUS.register(this);

        new CustomRotationMapping(new File(getMinecraftBaseDir(), "mods/resources/ruins"));
    }

    @EventHandler
    public void serverStarted(FMLServerStartingEvent evt)
    {
        evt.registerServerCommand(new CommandParseTemplate());
        evt.registerServerCommand(new CommandTestTemplate());
        evt.registerServerCommand(new CommandUndo());
    }

    private long nextInfoTime;

    @SubscribeEvent
    public void onBreakSpeed(BreakSpeed event)
    {
        ItemStack is = event.getEntityPlayer().func_184614_ca();
        if (is != null && is.func_77973_b() == Items.field_151055_y && System.currentTimeMillis() > nextInfoTime)
        {
            nextInfoTime = System.currentTimeMillis() + 1000L;
            event.getEntityPlayer().func_146105_b(new TextComponentTranslation(String.format("BlockName [%s], blockID [%s], metadata [%d]",
                    event.getState().func_177230_c().func_149732_F(), event.getState().func_177230_c().getRegistryName().func_110623_a(), event.getState().func_177230_c().func_176201_c(
                            event.getState()))));
        }
    }

    @SubscribeEvent
    public void onBreak(BreakEvent event)
    {
        if (event.getPlayer() != null && !(event.getPlayer() instanceof FakePlayer))
        {
            ItemStack is = event.getPlayer().func_184614_ca();
            if (is != null && is.func_77973_b() == Items.field_151055_y && System.currentTimeMillis() > nextInfoTime)
            {
                nextInfoTime = System.currentTimeMillis() + 1000L;
                event.getPlayer().func_146105_b(
                        new TextComponentTranslation(String.format("BlockName [%s], blockID [%s], metadata [%d]", event.getState().func_177230_c().func_149732_F(),
                                event.getState().func_177230_c().getRegistryName().func_110623_a(), event.getState().func_177230_c().func_176201_c(event.getState()))));
                event.setCanceled(true);
            }
        }
    }

    @SubscribeEvent
    public void eventWorldSave(WorldEvent.Save evt)
    {
        WorldHandle wh = getWorldHandle(evt.getWorld());
        if (wh != null)
        {
            wh.generator.flushPosFile(evt.getWorld().func_72912_H().func_76065_j());
        }
    }

    @SubscribeEvent
    public void onEntityEnteringChunk(EntityEvent.EnteringChunk event)
    {
        if (event.getEntity() instanceof EntityPlayer && !event.getEntity().field_70170_p.field_72995_K)
        {
            TileEntityCommandBlock tecb;
            ArrayList<TileEntityCommandBlock> tecblist = new ArrayList<>();

            for (int xoffset = -4; xoffset <= 4; xoffset++)
            {
                for (int zoffset = -4; zoffset <= 4; zoffset++)
                {
                    for (TileEntity teo : event.getEntity().field_70170_p.func_72964_e(event.getNewChunkX() + xoffset, event.getNewChunkZ() + zoffset).func_177434_r().values())
                    {
                        if (teo instanceof TileEntityCommandBlock)
                        {
                            tecb = (TileEntityCommandBlock) teo;
                            if (tecb.func_145993_a().func_145753_i().startsWith("RUINSTRIGGER "))
                            {
                                // strip prefix from command
                                tecb.func_145993_a().func_145752_a((tecb.func_145993_a().func_145753_i()).substring(13));
                                tecblist.add(tecb);
                            }
                        }
                    }
                }
            }

            for (TileEntityCommandBlock tecb2 : tecblist)
            {
                // call command block execution
                tecb2.func_145993_a().func_145755_a(event.getEntity().field_70170_p);
                // kill block
                BlockPos pos = tecb2.func_174877_v();
                System.out.printf("Ruins executed and killed Command Block at [%s]\n", pos);
                event.getEntity().field_70170_p.func_175698_g(pos);
            }
        }
    }

    private class RuinsWorldGenerator implements IWorldGenerator
    {
        @Override
        public void generate(Random random, int chunkX, int chunkZ, World world, IChunkGenerator chunkGenerator, IChunkProvider chunkProvider)
        {
            if (world.field_72995_K || !world.func_72912_H().func_76089_r())
            {
                return;
            }

            int[] tuple = { chunkX, chunkZ };
            if (currentlyGenerating.contains(tuple))
            {
                System.err.printf("Ruins Mod caught recursive generator call at chunk [%d|%d]", chunkX, chunkZ);
            }
            else
            {
                WorldHandle wh = getWorldHandle(world);
                if (wh != null)
                {
                    if (wh.fileHandle.allowsDimension(world.field_73011_w.getDimension()) && !getWorldHandle(world).chunkLogger.catchChunkBug(chunkX, chunkZ))
                    {
                        currentlyGenerating.add(tuple);
                        if (world.field_73011_w instanceof WorldProviderHell)
                        {
                            generateNether(world, random, tuple[0] * 16, tuple[1] * 16);
                        }
                        else
                        // normal world
                        {
                            generateSurface(world, random, tuple[0] * 16, tuple[1] * 16);
                        }
                        currentlyGenerating.remove(tuple);
                    }
                }
            }
        }
    }

    private void generateNether(World world, Random random, int chunkX, int chunkZ)
    {
        WorldHandle wh = getWorldHandle(world);
        if (wh.fileHandle != null)
        {
            while (!wh.fileHandle.loaded)
            {
                Thread.yield();
            }
            wh.generator.generateNether(world, random, chunkX, chunkZ);
        }
    }

    private void generateSurface(World world, Random random, int chunkX, int chunkZ)
    {
        WorldHandle wh = getWorldHandle(world);
        if (wh.fileHandle != null)
        {
            while (!wh.fileHandle.loaded)
            {
                Thread.yield();
            }
            wh.generator.generateNormal(world, random, chunkX, chunkZ);
        }
    }

    private class WorldHandle
    {
        FileHandler fileHandle;
        RuinGenerator generator;
        ChunkLoggerData chunkLogger;
    }

    private WorldHandle getWorldHandle(World world)
    {
        WorldHandle wh = null;
        if (!world.field_72995_K)
        {
            if (!generatorMap.containsKey(world.field_73011_w.getDimension()))
            {
                wh = new WorldHandle();
                initWorldHandle(wh, world);
                generatorMap.put(world.field_73011_w.getDimension(), wh);
            }
            else
            {
                wh = generatorMap.get(world.field_73011_w.getDimension());
            }
        }

        return wh;
    }

    private static File getWorldSaveDir(World world)
    {
        ISaveHandler worldsaver = world.func_72860_G();

        if (worldsaver.func_75763_a(world.field_73011_w) instanceof AnvilChunkLoader)
        {
            AnvilChunkLoader loader = (AnvilChunkLoader) worldsaver.func_75763_a(world.field_73011_w);

            for (Field f : loader.getClass().getDeclaredFields())
            {
                if (f.getType().equals(File.class))
                {
                    try
                    {
                        f.setAccessible(true);
                        // System.out.println("Ruins mod determines World Save Dir to be at: "+saveLoc);
                        return (File) f.get(loader);
                    }
                    catch (Exception e)
                    {
                        System.out.println("Ruins mod failed trying to find World Save dir:");
                        e.printStackTrace();
                    }
                }
            }
        }

        return null;
    }

    public static File getMinecraftBaseDir()
    {
        if (FMLCommonHandler.instance().getSide() == Side.CLIENT)
        {
            File file = FMLClientHandler.instance().getClient().field_71412_D;
            String abspath = file.getAbsolutePath();
            if (abspath.endsWith("."))
            {
                file = new File(abspath.substring(0, abspath.length() - 1));
            }
            return file;
        }
        return FMLCommonHandler.instance().getMinecraftServerInstance().func_71209_f("");
    }

    private void initWorldHandle(WorldHandle worldHandle, World world)
    {
        // load in defaults
        try
        {
            File worlddir = getWorldSaveDir(world);
            worldHandle.fileHandle = new FileHandler(worlddir, world.field_73011_w.getDimension());
            worldHandle.generator = new RuinGenerator(worldHandle.fileHandle, world);

            worldHandle.chunkLogger = (ChunkLoggerData) world.getPerWorldStorage().func_75742_a(ChunkLoggerData.class, "ruinschunklogger");
            if (worldHandle.chunkLogger == null)
            {
                worldHandle.chunkLogger = new ChunkLoggerData("ruinschunklogger");
                world.getPerWorldStorage().func_75745_a("ruinschunklogger", worldHandle.chunkLogger);
            }
        }
        catch (Exception e)
        {
            System.err.println("There was a problem loading the ruins mod:");
            System.err.println(e.getMessage());
            e.printStackTrace();
        }
    }

}
